/*************************************************************
*  This file is part of the Surface Evolver source code.     *
*  Programmer:  Ken Brakke, brakke@geom.umn.edu              *
*************************************************************/

/********************************************************************
*
*  File: storage.cpp
*
*  Purpose:  File defining details of storage implementation.
*            All machine-dependent gory details should be here
*            (purely private details) or in storage.h
*            (for inclusion in other source files that need to know).
*            
*      This version has element ids implemented as pointers with
*      low bits used as flags.
*/

#include "include.h"

/* unified block references, indexed by element type */
char *base[NUMELEMENTS];


struct web web = {
{ { VERTEX, NULL, 0, NULLID, NULLID, 0, NULLID },
 { EDGE  , NULL, 0, NULLID, NULLID, 0, NULLID },
 { FACET , NULL, 0, NULLID, NULLID, 0, NULLID },
 { BODY  , NULL, 0, NULLID, NULLID, 0, NULLID },
 { FACETEDGE, NULL, 0, NULLID, NULLID, 0, NULLID } },
 SOAPFILM,LINEAR
    };


element_id
  NULLVERTEX  =  NULL, 
  NULLEDGE    = NULL,
  NULLFACET   = NULL,
  NULLBODY    = NULL,  
  NULLFACETEDGE =  NULL;

void extend(type,number)
int type,number;
{ /* stub only */
}

element_id new_element(type)
int type;
{
  element_id newid;
  struct element *new_el,*last;

  switch ( type )
    {
      case VERTEX:   new_el = (element *)new vertex; break;
      case EDGE  :   new_el = (element *)new edge  ; break;
      case FACET :   new_el = (element *)new facet ; break;
      case BODY  :   new_el = (element *)new body  ; break;
      case FACETEDGE:   new_el = (element *)new facetedge; break;
    }
  if ( new_el == NULL )
    error("Cannot allocate element structure.\n",RECOVERABLE);
  memset(new_el,0,sizes[type]);  /* new does'nt zero out memory */
  
  newid = (element_id)new_el;

  /* add to end of in-use chain */
  new_el->forechain = NULLID;
  new_el->backchain = web.skel[type].last;
  if ( valid_id(web.skel[type].last) )
    {
      last = elptr(web.skel[type].last);    /* find end of in-use chain */
      last->forechain = newid;
    }
  else
    {      
      web.skel[type].used = newid;
    }
  web.skel[type].last = newid;

  new_el->attr = ALLOCATED | NEWELEMENT;
  new_el->ordinal = web.skel[type].max_ord++;
  if ( web.skel[type].max_ord > web.skel[type].maxcount )
     web.skel[type].maxcount = web.skel[type].max_ord;
  new_el->type = type;
  web.skel[type].count++;  
  return newid;
}

void free_element(id)
element_id id;
{
  struct element *ptr;
  int type = id_type(id);

/* printf("freeing %X  type %d ordinal %d\n",id,type,ordinal(id)); */

  if ( !valid_id(id) )
   {
     sprintf(errmsg,"Trying to free invalid id %08X \n",id);
     error(errmsg,RECOVERABLE);
   }
   
  ptr = elptr(id);
  if ( !(ptr->attr & ALLOCATED) )
   {
     sprintf(errmsg,"Trying to free unallocated element id %08X \n",id);
     error(errmsg,RECOVERABLE);
   }
  ptr->attr = 0;  /* clear attributes */

  /* remove from in-use list */
  if ( valid_id(ptr->forechain) )
    elptr(ptr->forechain)->backchain = ptr->backchain;
  else
    web.skel[type].last = ptr->backchain;

  if ( valid_id(ptr->backchain) )
    elptr(ptr->backchain)->forechain = ptr->forechain;
  else
    web.skel[type].used = ptr->forechain;

  /* add to discard list */
  /* save forechain for generators */
  ptr->backchain = web.skel[type].discard;
  web.skel[type].discard = id;

  web.skel[type].count--; 

}

void free_discards()
{ int type;

  for ( type = 0 ; type < NUMELEMENTS ; type++ )
    { element_id id = web.skel[type].discard;
      while ( valid_id(id) )
	{ struct element *ptr = elptr(id);
	  web.skel[type].discard = ptr->backchain;
          delete ptr;
	  id = web.skel[type].discard;
	}
    }
}

int ordinal(id)
element_id id;
{
  if ( !valid_id(id) ) return 0;
  return  elptr(id)->ordinal;
}

int generate_all(type,idptr)  /* re-entrant */
int type;
element_id *idptr;
{
  struct element *ptr;

  if ( !valid_id(*idptr) ) /* first time */
    { *idptr = web.skel[type].used;
      if ( !valid_id(*idptr) ) return 0;
    }
  else
    { ptr = elptr(*idptr);
      do
    	{ /* may have to get off discard list */
          *idptr = ptr->forechain;
          if ( !valid_id(*idptr) ) return 0;
          ptr = elptr(*idptr);
         }
      while ( !(ptr->attr & ALLOCATED) );
    }

  return 1;
}

void memory_report()
{
   long mem = 0;
   int k;

   mem = 0;
   for ( k = 0 ; k < NUMELEMENTS ; k++ )
     mem += web.skel[k].count*sizes[k];

   sprintf(msg,"Vertices: %ld   Edges: %ld   Facets: %ld   Facetedges: %ld   Memory: %ld\n",
     web.skel[0].count,web.skel[1].count,web.skel[2].count,web.skel[4].count,
     mem);
   outstring(msg);
}

void reset_skeleton()
{
  int i,j;
  struct boundary *bdry;
  struct constraint *con;
  struct surf_energy *surfen;
  struct quantity *quan;
  ETYPE type;

  if ( gauss1Dpt ) { free((char *)gauss1Dpt); gauss1Dpt = NULL; }
  if ( gauss1Dwt ) { free((char *)gauss1Dwt); gauss1Dwt = NULL; }

  /* deallocate boundary specs */
  for ( i = 0, bdry = web.boundaries ; i < BDRYMAX ; i++, bdry++ )
    if ( bdry->coordf[0] )
      {
        for ( j = 0 ; j < web.sdim ; j++ )
	  free_expr(bdry->coordf[j]);
	free((char *)bdry->coordf[0]);
      }

  /* deallocate constraint specs */
  for ( i = 0, con = web.constraints ; i < CONSTRMAX ; i++, con++ )
    constraint_free(con);

  /* deallocate surface energy specs */
  for ( i = 0, surfen = web.surfen ; i < web.surfen_count ; i++, surfen++ )
    { if ( surfen->envect[0] )
        { for ( j = 0 ; j < web.sdim ; j++ )
             free_expr(surfen->envect[j]);
          free((char *)surfen->envect[0]);
	}
    }

  /* deallocate quantity specs */
  for ( i = 0, quan = web.quants ; i < web.quantity_count ; i++, quan++ )
    { if ( quan->attr & IN_USE )
        { for ( j = 0 ; j < web.sdim ; j++ )
             free_expr(quan->quanvect[j]);
          free((char *)quan->quanvect[0]);
	}
    }

  /* deallocate metric expressions */
  if ( web.metric_flag )
    for ( i = 0 ; i < web.sdim ; i++ )
      for ( j = 0 ; j < web.sdim ; j++ )
	free_expr(&web.metric[i][j]);

  /* free parameters */
  if ( web.params ) free((char *)web.params);

#ifdef TC
if ( farheapcheck() < 0 )
  error("Corrupt heap.\n",UNRECOVERABLE);
#endif

  for ( type = 0 ; type < NUMELEMENTS ; type++ )
    {
      /* patch used list into discard list */
      elptr(web.skel[type].used)->backchain = web.skel[type].discard;
      web.skel[type].discard = web.skel[type].last;
      web.skel[type].used = NULLID;
      web.skel[type].last = NULLID;
    }
  free_discards();

#ifdef TC
if ( farheapcheck() < 0 )
  error("Corrupt heap.\n",UNRECOVERABLE);
#endif

  if ( web.inverse_periods )  free_matrix(web.inverse_periods);

  memset((char *)&web,0,sizeof(web));  /* total clean out */
}

/*******************************************************************
*
*  function: vgrad_init()
*
*  purpose:  allocates storage for vertex volume gradients
*
*/

volgrad huge *vgradbase;   /* allocated list of structures */
volgrad * huge *vgptrbase;   /* allocated list of chain start pointers */
int      vgradtop;    /* number of first free  structure */
long     vgradmax;    /* number allocated */

void vgrad_init(qfixed)
int qfixed; /* how many fixed quantities */
{
  long allocsize;
  int i ;

  allocsize = (long)(web.skel[VERTEX].max_ord+1)*sizeof(volgrad huge *);
  if ( allocsize < MAXALLOC )
   vgptrbase = (volgrad * huge *)temp_calloc(web.skel[VERTEX].max_ord+1,
                                                   sizeof(volgrad huge *));
  if ( vgptrbase == NULL )
    {
      sprintf(errmsg,"Cannot allocate %d volgrad pointers of size %d\n",
        web.skel[VERTEX].max_ord+1, sizeof(volgrad huge *));
      error(errmsg,RECOVERABLE);
    }

  if ( web.simplex_flag )  /* may be gross overestimate, but safe */
    vgradmax = web.skel[VERTEX].count*web.skel[BODY].count;
  else if ( web.dimension == STRING )
    vgradmax = 2*web.skel[EDGE].count;
  else /* SOAPFILM */
    /* could have factor of 2 on facets if could guarantee bodies
       next to an edge are separated by a facet, which we can't
       if we permit incomplete bodies. 
     */
    vgradmax = web.skel[FACETEDGE].count - web.skel[FACET].count
             + web.skel[BODY].count + qfixed*web.skel[VERTEX].count + 10;

  if ( web.modeltype == QUADRATIC )
    vgradmax += web.skel[FACETEDGE].count;  /* for midpoints */

  allocsize = (long)vgradmax*sizeof(struct volgrad);
  vgradbase = (struct volgrad huge *)farmalloc((long)vgradmax*sizeof(struct volgrad));
  if ( vgradbase == NULL )
    {
      sprintf(errmsg,"Cannot allocate %ld volgrad structures of size %d\n",
        vgradmax,sizeof(struct volgrad));
      printf("farcoreleft: %ld\n",farcoreleft());
      vgrad_end();
      error(errmsg,RECOVERABLE);
    }
  for ( i = 0 ; i < vgradmax ; i++ )
    memset(vgradbase+i,0,sizeof(struct volgrad));
  
  vgradtop = 0;

}

void vgrad_end()
{
  if ( vgradbase ) { farfree((char *)vgradbase); vgradbase = NULL; }
  if ( vgptrbase ) { temp_free((char *)vgptrbase); vgptrbase = NULL; }
  vgradmax = 0;
}

volgrad *get_vertex_vgrad(vertex_id v_id)
{
  return (volgrad far*)vgptrbase[ordinal(v_id)];
}

volgrad huge *new_vgrad()
{

  if ( vgradtop >= vgradmax ) 
  /* error */
    {
      vgrad_end();
      sprintf(errmsg,"Too many volgrad structures requested: %d\n", vgradtop);
      error(errmsg,RECOVERABLE);
    }

  return vgradbase + vgradtop++;
}

      
volgrad *get_bv_vgrad(body_id b_id,vertex_id v_id)
{
  volgrad huge *vgptr;

  vgptr = get_vertex_vgrad(v_id);
  while ( vgptr )
   if ( vgptr->b_id == b_id ) break;
   else vgptr = vgptr->chain;

  return (volgrad far *)vgptr;   /* null if not found */
}
      
volgrad *get_bv_new_vgrad(body_id b_id,vertex_id v_id)
{
  volgrad  * huge *ptrptr;  /* pointer to pointer so can update if need be */

  ptrptr = vgptrbase + ordinal(v_id);
  while ( *ptrptr )
   if ( (*ptrptr)->b_id == b_id ) return *ptrptr;
   else ptrptr = &(*ptrptr)->chain;

  /* need to get new structure */
  *ptrptr = new_vgrad();
  (*ptrptr)->b_id = b_id;

  return (volgrad far *)*ptrptr;
}
