/*************************************************************
*  This file is part of the Surface Evolver source code.     *
*  Programmer:  Ken Brakke, brakke@geom.umn.edu              *
*************************************************************/


/*************************************************************
*
*   file:      calcforc.c
*
*   Contents:  Functions calculating energy, volume, and 
*              their gradients at vertices.
*/

#include "include.h"


/****************************************************************
*
*  Function: calc_force()
*
*  Purpose: calculates net force at triangulation vertices due
*           to surface tension and constraints
*
*  
*/

void calc_force()  
{
  vertex_id v_id;
  facet_id f_id;
  edge_id e_id;
  REAL **a;
  int i,j;

  /* zero out vertex cumulative quantities */
  FOR_ALL_VERTICES(v_id)
    { 
      REAL *f = get_force(v_id);
      for ( i = 0 ; i < web.sdim ; i ++ ) f[i] = 0.0;
      set_vertex_star(v_id,0.0);          
      set_vertex_valence(v_id,0);
    }
  FOR_ALL_EDGES(e_id)
    set_edge_star(e_id,0.0);

  if ( square_curvature_flag && !kusner_flag ) sqcurve_force_init();

  if ( unit_normal_flag )  /* for Dennis DeTurck */
    {  
      FOR_ALL_VERTICES(v_id)
        { REAL *f = get_force(v_id);
	  facetedge_id fe = get_vertex_fe(v_id);
	  if ( inverted(get_fe_facet(fe)) ) fe = get_next_facet(fe);
          calc_vertex_normal(fe,f);
	  for ( i = 0 ; i < web.sdim ; i++ )
	     f[i] *= deturck_factor;
	}
      /* continue with other forces */
    }

  /* boundary and constraint forces */
  if ( web.simplex_flag )
    {
      FOR_ALL_EDGES(e_id)
	calc_simplex_edge_force(e_id);
    }
  else if ( web.dimension == STRING )
    {
      FOR_ALL_VERTICES(v_id)
       { ATTR attr = get_vattr(v_id);
	 if ( attr &  FIXED ) continue;
	 if ( square_curvature_flag && !kusner_flag ) 
	   sqcurve_force_string(v_id);
         if ( !(attr & BDRY_ENERGY) ) continue;
         if ( attr & BOUNDARY )
           calc_bdry_force_v(v_id);
         else if ( attr & CONSTRAINT )
           calc_constr_force_v(v_id);
       }
    }
  else /* SOAPFILM */
    {
      FOR_ALL_EDGES(e_id)
       { ATTR attr = get_eattr(e_id);
	 if ( attr &  FIXED ) continue;
         if ( attr & DENSITY )
           (*calc_edge_forces)(e_id);  /* for triple line energies */
         if ( !(attr & BDRY_ENERGY) ) continue;
         if ( attr & BOUNDARY )
           calc_bdry_force_e(e_id);
         else if ( attr & CONSTRAINT )
           calc_constr_force_e(e_id);
       }
    }

  if ( web.simplex_flag )
    {
      FOR_ALL_FACETS(f_id)
	calc_simplex_forces(f_id);
    }
  else if ( web.dimension == STRING )
    {
      /* tension forces */
      /* add each edge's contribution to its endpoints */
      FOR_ALL_EDGES(e_id)
       { ATTR attr = get_eattr(e_id);
         if ( !(attr & FIXED) )
         (*calc_edge_forces)(e_id);
         if ( attr & BDRY_ENERGY )
               calc_constr_force_e(e_id);  /* substitute surface energy */
       }
    }
  else /* get here only for SOAPFILM */
    {
      /* tension forces */
      /* find each triangle's contribution to forces on its vertices */
      FOR_ALL_FACETS(f_id)
       if ( !(get_fattr(f_id) & FIXED) )
        (*calc_facet_forces)(f_id);
    }

  if ( square_curvature_flag && !kusner_flag ) sqcurve_force_end();

  if ( square_curvature_flag && kusner_flag ) kusner_force();

  /* Add kludge forces on boundary edges to prevent pulling away */
  if ( (web.modeltype == LINEAR) & web.convex_flag )
    {
      FOR_ALL_EDGES(e_id)
        {
          if ( !(get_eattr(e_id) & FIXED) )
          bdry_force(e_id);
          constr_springs(e_id);
        }
    }

  /* For quadratic film model, need to add some adjacent midpoint motion
     to vertices to get proper motion */
  if ( web.dimension == SOAPFILM )
      if ( web.modeltype == QUADRATIC )
        { edge_id e_id;

          FOR_ALL_EDGES(e_id)
            {
              vertex_id headv,tailv;
              REAL *fh,*ft,*force;
    
              if ( get_eattr(e_id) & FIXED ) continue;
              force = get_force(get_edge_midv(e_id));
    
              headv = get_edge_headv(e_id);
              fh = get_force(headv);
              tailv = get_edge_tailv(e_id);
              ft = get_force(tailv);

              for ( i = 0 ; i < web.sdim ; i++ )
                { fh[i] += 0.5*force[i];
                  ft[i] += 0.5*force[i];
                }
            }
        }

  /* project to constraints, etc */
  fix_vertices();

  /* do area normalization */
  if ( web.area_norm_flag ) 
    { 
      FOR_ALL_VERTICES(v_id)
        {
          REAL area = get_vertex_star(v_id)/star_fraction;
          REAL *force = get_force(v_id);
	  REAL ff;
	  if ( get_vattr(v_id) & FIXED ) continue;
	  if ( effective_area_flag && (web.dimension == STRING) )
	    { /* calculate effective area */
	      double d;
	      int valence = get_vertex_valence(v_id);
	      ff = dot(force,force,web.sdim);
	      if ( ff == 0.0 ) continue;
	      d = get_edge_density(get_fe_edge(get_vertex_fe(v_id)));
	      ff /= d*d;
	      if ( valence == 2 )
	 	{ double f2;
		  f2 = sqrt(ff)/2;
		  area *= sqrt(1 - ff/4)*f2/asin(f2);
                }
              else if ( valence == 1 )
		{ if ( !(get_vattr(v_id) & (FIXED|CONSTRAINT|BOUNDARY) ) )
		    { edge_id e_id = get_fe_edge(get_vertex_fe(v_id));
		      vertex_id other_v = get_edge_headv(e_id);
		      eliminate_edge(e_id);
		      free_element(e_id);
                      add_vertex_valence(other_v,-1);
		      continue;
		    }
		  area *= sqrt(1 - ff);
		}
              else if ( valence == 0 )
		area = 1.0;  /* disconnected pt; no force anyway */
              else if ( (valence == 3) && !old_area_flag )
		{ /* for density 1 edges only */
		  facetedge_id fe = get_vertex_fe(v_id);
		  facetedge_id next_fe = fe;
		  edge_id e_id;
		  double ss[3],f[MAXCOORD],side[3][MAXCOORD];
		  double ang12,ang13,det,leg[2][MAXCOORD];
		  area = 0.0;
		  for ( i = 0 ; i < 3 ; i++ )
		    { e_id = get_fe_edge(next_fe);
		      ss[i] = get_edge_length(e_id);
		      get_edge_side(e_id,side[i]);
		      next_fe = get_prev_edge(get_next_facet(next_fe));
		      invert(next_fe);
                    }
		  ang12 = acos(dot(side[0],side[1],web.sdim)/ss[0]/ss[1])
			   - 2*M_PI/3;
		  ang13 = acos(dot(side[0],side[2],web.sdim)/ss[0]/ss[2])
			   - 2*M_PI/3;
		  for ( j = 0 ; j < web.sdim ; j++ )
		   { leg[0][j] = side[1][j] - side[0][j];
		     leg[1][j] = side[0][j] - side[2][j];
		   } 
		  det = fabs(leg[0][0]*leg[1][1] - leg[0][1]*leg[1][0]);
		  f[0] = (leg[1][0]*ang12 - leg[0][0]*ang13)/det;
		  f[1] = (leg[1][1]*ang12 - leg[0][1]*ang13)/det;
		  area = -0.5*dot(f,force,2)/dot(f,f,2);
                  set_vertex_star(v_id,star_fraction*area);
		  for ( i = 0 ; i < 2 ; i++ )
		    force[i] = -2*f[i];
                  continue;
                }
	      else /* triple point at least */
		{ facetedge_id fe = get_vertex_fe(v_id);
		  facetedge_id next_fe = fe;
		  edge_id e_id;
		  double ss,fs,side[MAXCOORD];
		  area = 0.0;
		  do
		    { e_id = get_fe_edge(next_fe);
		      get_edge_side(e_id,side);
		      ss = dot(side,side,web.sdim);
		      fs = dot(force,side,web.sdim);
		      ff = dot(force,force,web.sdim);
		      area += 0.5*sqrt(ff*ss - fs*fs);
		      next_fe = get_prev_edge(get_next_facet(next_fe));
		      invert(next_fe);
                    }
                  while ( !equal_id(fe,next_fe) );
                }
              set_vertex_star(v_id,star_fraction*area);
            }
          if ( area == 0.0 )  
	    { sprintf(errmsg,"Zero area around vertex %d.\n",ordinal(v_id)+1);
	      error(errmsg,RECOVERABLE);
            }
          for ( i = 0 ; i < web.sdim ; i++ )
            force[i] /= area;
        }
    }

  /* project to parameter space for boundary points */
  a = dmatrix(0,2,0,2);
  FOR_ALL_VERTICES(v_id)
    {
      int pcount;
      REAL *f = get_force(v_id);
      REAL tmp[MAXCOORD];
      struct boundary *bdry;
      int m;

      if ( get_vattr(v_id) & FIXED ) continue;
      if ( !(get_vattr(v_id) & BOUNDARY) ) continue;
      bdry = get_boundary(v_id);
      pcount = bdry->pcount;
      b_proj(bdry,get_param(v_id),a,PARAMPROJ);
      matvec_mul(a,f,tmp,pcount,web.sdim);
      for ( m = 0 ; m < pcount ; m++ ) f[m] = tmp[m];
      for ( m = pcount ; m < web.sdim ; m++ ) f[m] = 0.0;
    }
  free_matrix(a);

  return;
}

/*******************************************************************
*
*  Function: calc_energy()
*
*  Purpose: Finds total energy of configuration.
*           Requires volumes set to calculate pressure energy.
*
*/

void calc_energy() 
{
  body_id   b_id;
  edge_id e_id;
  int n;

  web.total_energy = 0.0;
  web.total_area   = 0.0;
  web.spring_energy = 0.0;
  euclidean_area = 0.0;

  /* zero out quantities */
  for ( n = 0 ; n < web.quantity_count ; n++ )
    web.quants[n].value = 0.0;

  if ( square_curvature_flag  && !kusner_flag ) sqcurve_energy_init();

  if ( web.simplex_flag )
    { facet_id f_id;
      FOR_ALL_FACETS(f_id)
       if ( !(get_fattr(f_id) & FIXED) )
	calc_simplex_energy(f_id);
      FOR_ALL_EDGES(e_id)
	calc_simplex_edge_energy(e_id);
    }
  else if ( web.dimension == STRING )
    {
      vertex_id v_id;
         
      FOR_ALL_EDGES(e_id)
       { ATTR attr = get_eattr(e_id);
       
	 if ( attr & FIXED ) continue;
         (*calc_edge_energy)(e_id);
         if ( attr & BDRY_ENERGY )
               calc_constr_energy_e(e_id);  /* substitute surface energy */
       }

      /* boundary energy  and square curvature */
      FOR_ALL_VERTICES(v_id)
       { ATTR attr = get_vattr(v_id);
	 if ( attr & FIXED ) continue;
	 if ( square_curvature_flag && !kusner_flag  ) 
	   sqcurve_energy_string(v_id);
         if ( attr & BDRY_ENERGY )
           {
             if ( attr & BOUNDARY )
               calc_bdry_energy_v(v_id);
             else if ( attr & CONSTRAINT )
               calc_constr_energy_v(v_id);
            }
        }
    }

  else /* web.dimension == SOAPFILM */
    {   
      facet_id  f_id;
      edge_id   e_id;

      FOR_ALL_FACETS(f_id)
       if ( !(get_fattr(f_id) & FIXED ) )
        (*calc_facet_energy)(f_id);

      FOR_ALL_EDGES(e_id)
       { ATTR attr = get_eattr(e_id);
	 if ( attr & FIXED ) continue;
         if ( attr & BDRY_ENERGY ) 
           {
             if ( attr & BOUNDARY )
               calc_bdry_energy_e(e_id);
             if ( attr & CONSTRAINT )
               calc_constr_energy_e(e_id);
           }
         if ( attr & DENSITY )
           (*calc_edge_energy)(e_id);  /* for triple line energies */
       }
    }

  if ( square_curvature_flag  && !kusner_flag ) sqcurve_energy_end();

  if ( square_curvature_flag && kusner_flag ) kusner_energy();

  if ( sqgauss_flag ) sqgauss_energy();

  /* Add kludge forces on boundary edges to prevent pulling away */
  if ( web.modeltype == LINEAR )
    {
      edge_id e_id;

      FOR_ALL_EDGES(e_id)
        {
	  if ( get_eattr(e_id) & FIXED ) continue;
          bdry_spring_energy(e_id);
          constr_spring_energy(e_id);
        }
    }

   FOR_ALL_BODIES(b_id)
     { REAL fix,vol;

       fix = get_body_fixvol(b_id);
       vol = get_body_volume(b_id);
       if ( web.pressure_flag )
         {
           if ( !equal_id(b_id,web.outside_body) )
             { web.total_energy += -web.pressure*fix*log(vol/fix);
               if ( valid_id(web.outside_body) )
                 web.total_energy += web.pressure*vol;
             }
         }
       else if ( get_battr(b_id) & PRESSURE )
         web.total_energy -= get_body_pressure(b_id)*vol;
     }

  extrap_val[level] = web.total_energy;
}

/*************************************************************
*
*  Function: calc_content()
*
*  Purpose: calculates body volumes 
*/

void calc_content()  
{
  facet_id  f_id;
  body_id   b_id;
  facetedge_id fe_id;

  if ( web.bodycount == 0 ) return;

  /* starting body volumes */
  FOR_ALL_BODIES(b_id)
    set_body_volume(b_id,get_body_volconst(b_id));

  if ( web.simplex_flag )
    { FOR_ALL_FACETS(f_id)
        calc_simplex_volume(f_id);
    }
  else if ( web.dimension == STRING ) 
    {
      vertex_id v_id;

       FOR_ALL_FACETEDGES(fe_id)
         (*calc_edge_area)(fe_id);

      FOR_ALL_VERTICES(v_id)
       { ATTR attr = get_vattr(v_id);
         if ( !(attr & BDRY_CONTENT) ) continue;
         if ( attr & BOUNDARY )
           calc_bdry_content_v(v_id);
         else if ( attr & CONSTRAINT )
           calc_constr_content_v(v_id);
       }
    }

  else if ( web.dimension == SOAPFILM ) 
   if ( web.torus_flag ) torvol();
   else
    {
      edge_id e_id;

      FOR_ALL_FACETS(f_id)
        (*calc_facet_volume)(f_id);

      FOR_ALL_EDGES(e_id)
       { ATTR attr = get_eattr(e_id);
         if ( !(attr & BDRY_CONTENT) ) continue;
         if ( attr & BOUNDARY )
           calc_bdry_content_e(e_id);
         else if ( attr & CONSTRAINT )
           calc_constr_content_e(e_id);
       }
    }

  web.vol_flag = 1;
}

/*************************************************************
*
*  Function: calc_pressure()
*
*  Purpose: calculates body volumes and pressures on faces 
*/

void calc_pressure()  
{
  body_id   b_id;

  if ( !web.pressure_flag ) return;

  /* now compute pressure in each body */
  FOR_ALL_BODIES(b_id)
    { 
      REAL p;
  
      if ( equal_id(b_id,web.outside_body) )
        p = web.pressure;
      else if ( get_battr(b_id) & FIXEDVOL )
        if ( get_body_volume(b_id) > 0.0 )
          p = web.pressure*get_body_fixvol(b_id)/get_body_volume(b_id);
        else 
          { sprintf(errmsg,"Body %d has volume %f\n",1+ordinal(b_id),
                                                get_body_volume(b_id));
            error(errmsg,WARNING);
            p = 0.0;
          }
      else
       { sprintf(errmsg,"Body %d has no prescribed volume %f\n",
                       1+ordinal(b_id), get_body_volume(b_id));
         error(errmsg,WARNING);
         p = 0.0;
       }

      set_body_pressure(b_id,p);
    }

}
