// viewchanger.C

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993, 1994 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       December 13, 1994
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/


#ifdef __GNUG__
#pragma implementation
#endif

#include "application.h"
#include "viewchanger.h"
#include "controller.h"
#include "query_templates.h"
#include "editor.h"
#include "dataview.h"
#include "data.h"
#include "valuerequester.h"
#include "smpte.h"

template <class T>
class MinMaxRequester : public TitledRequester {
public:
	MinMaxRequester(const char* title,
	                const char* minlabel, T& minval, const Range &,
	                const char* maxlabel, T& maxval, const Range &);
protected:
	redefined void configureRequest(Request *);
protected:
	const char* minLabel;
	const char* maxLabel;
	T &minVal;
	T &maxVal;
	const Range &minRange;
	const Range &maxRange;
};

template <class T>
MinMaxRequester<T>::MinMaxRequester(const char* title,
		const char* minlabel, T& minval, const Range& minrange,
		const char* maxlabel, T& maxval, const Range& maxrange)
	: TitledRequester(title), minLabel(minlabel), maxLabel(maxlabel),
	  minVal(minval), maxVal(maxval),
	  minRange(minrange), maxRange(maxrange) {}

template <class T>
void
MinMaxRequester<T>::configureRequest(Request* request) {
	request->appendValue(minLabel, &minVal, minRange);
	request->appendValue(maxLabel, &maxVal, maxRange);
}

//********

ViewChanger::ViewChanger(DataView* view) : myView(view) {
	myView->ref();
}

ViewChanger::~ViewChanger() { Resource::unref(myView); }

Data *
ViewChanger::viewData() { return myView->dataptr; }

int
ViewChanger::apply() {
	return doApply(myView);
}

//********

ScaleOptionsSetter::ScaleOptionsSetter(DataView* view)
	: ViewChanger(view), smpteFormat(SMPTE::currentFormat()) {}

Requester *
ScaleOptionsSetter::createRequester() {
	return new ChoiceRequester("Set Horizontal Scale Options:",
	                    "SMPTE Frame Format:",
	                    SMPTE::formatNames(),
	                    smpteFormat);
}

int
ScaleOptionsSetter::doApply(DataView* view) {
	SMPTE::setFormat(SMPTE::Format(smpteFormat));
	view->resetHorizontalScale();
	return true;
}

//********

VisibleFrameRange::VisibleFrameRange(DataView* view) : ViewChanger(view),
	  frameRange(view->getVisibleFrameRange()),
	  minFrame(frameRange.intMin()), maxFrame(frameRange.intMax()) {}

Requester *
VisibleFrameRange::createRequester() {
	if(myView->frameScaleInFrames()) {
		const Range startRange = viewData()->frameRange(FrameUnit);
		const Range endRange(1, startRange.intMax());
		return new MinMaxRequester<int>(
			"Set Endpoints for Frame Display:",
			"Start Frame:", minFrame, startRange,
			"End Frame:", maxFrame, endRange
		);
	}
	else {
		// convert frame range to time range,
		// i.e., frames (0 -> n-1) becomes (0 -> n) / samprate 
		frameRange.set(frameRange.min(), frameRange.max() + 1);
		frameRange *= 1.0/viewData()->frameRate();
		const Range timeRange = viewData()->frameRange(FrameTimeUnit);
		return new MinMaxRequester<double>(
			"Set Endpoints for Frame Display:",
			"Start Time:", frameRange.min_(), timeRange,
			"End Time:", frameRange.max_(), timeRange
		);
	}
}

int
VisibleFrameRange::doApply(DataView* view) {
	if(!view->frameScaleInFrames())
		frameRange *= viewData()->frameRate();
	else frameRange.set(minFrame, maxFrame);

	view->setVisibleFrameRange(frameRange);
	return true;
}

//********

class InsertPointRequester : public  MinMaxRequester<int> {
public:
	InsertPointRequester(int& point, int& minVal, int& maxVal);
protected:
	redefined void configureRequest(Request *);
private:
	int& thePoint;
};

InsertPointRequester::InsertPointRequester(int& point, int& minVal, int& maxVal)
	  : MinMaxRequester<int>(
	  	"Set Insert Point:",
	  	"First Channel:", minVal, NonNegativeIntegers,
	  	"Last Channel:", maxVal, NonNegativeIntegers
	  ),
	  thePoint(point) {}
	 
void
InsertPointRequester::configureRequest(Request *request) {
	request->appendValue("Frame:", &thePoint, NonNegativeIntegers);
	MinMaxRequester<int>::configureRequest(request);
}


//********

class InsertTimeRequester : public MinMaxRequester<int> {
public:
	InsertTimeRequester(float& time, int& minVal, int& maxVal);
protected:
	redefined void configureRequest(Request *);
private:
	float& theTime;
};

InsertTimeRequester::InsertTimeRequester(float& time, int& minVal, int& maxVal)
	  : MinMaxRequester<int>(
	  	"Set Insert Time",
	  	"First Channel:", minVal, NonNegativeIntegers,
	  	"Last Channel:", maxVal, NonNegativeIntegers
	  ),
	  theTime(time) {}

void
InsertTimeRequester::configureRequest(Request *request) {
	request->appendValue("Time:", &theTime, NonNegativeNumbers),
	MinMaxRequester<int>::configureRequest(request);
}

//********

InsertPoint::InsertPoint(DataView* view)
		: ViewChanger(view), insertPoint(0), insertTime(0) {
	Range channelRange = view->getChannelRange();
	minChan = channelRange.intMin();
	maxChan = channelRange.intMax();
}

Requester *
InsertPoint::createRequester() {
	if(myView->frameScaleInFrames())
		return new InsertPointRequester(insertPoint, minChan, maxChan);
	else
		return new InsertTimeRequester(insertTime, minChan, maxChan);
}

int
InsertPoint::doApply(DataView* view) {
	insertPoint = view->frameScaleInFrames() ? insertPoint :
		round(insertTime * viewData()->frameRate());
	int status = true;
	if(insertPoint > viewData()->length()) {
		Response r = Application::choice(
			"This insert point is beyond the end of the file.",
			nil, nil, No,
			"extend file", "adjust insert", "cancel");
		switch(r) {
		case Yes:	// extend it
			status = viewData()->changeLength(insertPoint + 1);
			if(status) viewData()->Notify();
			else Application::alert("Failed to extend file.",
			                        "Cancelling command.");
			break;
		case No:	// adjust insert
			insertPoint = viewData()->length() - 1;
			break;
		case Cancel:
		default:
			status = false;
		}
	}
	if(status)
		view->setInsertPoint(insertPoint, Range(minChan, maxChan));
	return status;
}

//********

class EditFramesRequester : public MinMaxRequester<int> {
public:
	EditFramesRequester(int&, int&, int&, int&, const Range&);
protected:
	redefined void configureRequest(Request *);
private:
	int& minFrame;
	int& maxFrame;
};

EditFramesRequester::EditFramesRequester(int& minframe, int& maxframe,
                                         int& startchan, int& endchan,
                                         const Range& channelRange)
	: MinMaxRequester<int>(
		"Set Edit Region:",
		"Start Channel:", startchan, channelRange,
		"End Channel:", endchan, channelRange),
		 minFrame(minframe), maxFrame(maxframe) {}

void
EditFramesRequester::configureRequest(Request *request) {
	request->appendValue("Start Frame:", &minFrame, NonNegativeIntegers);
	request->appendValue("End Frame:", &maxFrame, PositiveIntegers);
	MinMaxRequester<int>::configureRequest(request);
}

class EditTimeRequester : public MinMaxRequester<int> {
public:
	EditTimeRequester(double&, double&, int&, int&, const Range &);
protected:
	redefined void configureRequest(Request *);
private:
	double& minTime;
	double& maxTime;
};

EditTimeRequester::EditTimeRequester(
		double& mintime, double& maxtime,
		int& startchan, int& endchan,
		const Range& channelRange)
	: MinMaxRequester<int>(
		"Set Edit Region:",
		"Start Channel:", startchan, channelRange,
		"End Channel:", endchan, channelRange),
		 minTime(mintime), maxTime(maxtime) {}

void
EditTimeRequester::configureRequest(Request *request) {
	request->appendValue("Start Time:", &minTime, NonNegativeNumbers);
	request->appendValue("End Time:", &maxTime, PositiveNumbers);
	MinMaxRequester<int>::configureRequest(request);
}

//********

EditRegion::EditRegion(DataView* view)
		: ViewChanger(view),
		  inFrames(view->frameScaleInFrames()),
		  editRange(view->getVisibleFrameRange()) {

	if(inFrames) {
		minFrame = editRange.intMin();
		maxFrame = editRange.intMax();
	}
	else {
		// increase by one to cvt to time range from frame range
		editRange.set(editRange.intMin(), editRange.intMax() + 1);
		editRange *= 1.0/viewData()->frameRate();
	}
	
	Range channelRange = view->getChannelRange();
	minChan = channelRange.intMin();
	maxChan = channelRange.intMax();
}

Requester *
EditRegion::createRequester() {
	const Range chanRange = viewData()->channelRange();
	if(inFrames)
		return new EditFramesRequester(
			minFrame, maxFrame, minChan, maxChan, chanRange
		);
	else
		return new EditTimeRequester(
			editRange.min_(), editRange.max_(), minChan, maxChan, chanRange
		);
}

int
EditRegion::doApply(DataView* view) {
	int status = true;
	if(inFrames)
		editRange.set(minFrame, maxFrame);
	else {
		editRange *= viewData()->frameRate();
		// reduce by one to cvt from time range to frame range
		editRange.set(editRange.intMin(), editRange.intMax() - 1);
	}
	int len = viewData()->length();
	if(len <= editRange) {
		Response r = Application::choice(
			"This edit region extends beyond the end of the file.",
			nil, nil, No,
			"extend file", "adjust region", "cancel");
		switch(r) {
		case Yes:	// extend it
			status = viewData()->changeLength(editRange.intMax());
			if(status) viewData()->Notify();
			else Application::alert("Failed to extend file.",
			                        "Cancelling command.");
			break;
		case No:	// truncate region
			editRange.set(editRange.intMin(), len);
			break;
		case Cancel:
		default:
			status = false;
		}
	}
	if(status) {
		Range channelRange(minChan, maxChan);
		view->setEditRegion(editRange, channelRange);
	}
	return status;
}

//********

ChannelRange::ChannelRange(DataView* view, const Range& current)
		: ViewChanger(view), displayRange(current) {
	minVal = current.intMin();
	maxVal = current.intMax();
}

int
ChannelRange::doApply(DataView* view) {
	view->setChannelRange(displayRange);
	return true;
}

//********

ChannelChannelRange::ChannelChannelRange(DataView* view)
	: ChannelRange(view, view->getChannelRange()) {}

Requester *
ChannelChannelRange::createRequester() {
	const Range allChans = viewData()->channelRange();
	return new MinMaxRequester<int>(
		"Set Channel Display Range:",
		"First Channel:", minVal, allChans,
	   	"Last Channel:", maxVal, allChans);
}

void
ChannelChannelRange::initialize() {
	displayRange.set(minVal, maxVal);
	Super::initialize();
}

//********

FrameChannelRange::FrameChannelRange(DataView* view)
		: ChannelRange(view, view->getDisplayedHorizRange()) {
	maxFrames = displayRange.max();
}

Requester *
FrameChannelRange::createRequester() {
	if(myView->horizScaleInFrames()) {
		const Range allFrames = viewData()->frameRange();
		return new MinMaxRequester<int>(
				"Set Horizontal Display Range:",
				"Start:", minVal, allFrames,
				"End:", maxVal, allFrames
		);
	}
	else
		return new MinMaxRequester<double>(
				"Set Horizontal Display Range:",
				"Start:", displayRange.min_(), NonNegativeNumbers,
				"End:", displayRange.max_(), PositiveNumbers
		);
}

void
FrameChannelRange::initialize() {
	int totalFrames = viewData()->frameLength();
	boolean inIntegers = myView->horizScaleInFrames();
	double min = inIntegers ? minVal
		: round(totalFrames * displayRange.min()/maxFrames);
	double max = inIntegers ? maxVal
		: round(totalFrames * displayRange.max()/maxFrames);
	displayRange.set(min, max);
	Super::initialize();
}
