// cs_pvocheader.C

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993, 1994 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       December 13, 1994
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/

#ifdef __GNUG__
#pragma implementation
#endif

#include "localdefs.h"
#include "application.h"
#include "cs_pvocheader.h"
#include "datafile.h"

CS_PvocHeader::CS_PvocHeader(int chans, double frate, int srate) 
		: FrameDataHeader(PV_MAGIC, FloatData, chans, frate, srate) {
	data_offset = diskHeaderSize();
}

int
CS_PvocHeader::readInfo(DataFile *file) {
	DiskHeader head;
	int status = false;
	if(file->read((void *) &head, diskHeaderSize()).good()) {
		samprate = int(head.samplingRate);
		framerate = head.samplingRate / head.frameIncr;
		setDataOffset(head.headBsize);
		setDataSize(head.dataBsize);
		nchans = head.frameSize + 2;	// N+2 chans data
		data_type = (head.dataFormat == PVFLOAT) ? FloatData : NoData;
		status = true;
	}
	else
		Application::alert("readInfo:  file read error.");
	return status && checkHeader();
}

int
CS_PvocHeader::checkHeader() {
	char msg[64];
	msg[0] = '\0';	// null for later check
	int retcode = 0;
	if(nchans == 0 || nchans % 2 != 0)
		sprintf(msg, "Possibly corrupt header: nchans = %d", nchans);
	else if(data_type != FloatData)
		sprintf(msg,
			"MXV cannot handle non-floating-point pvoc files at this time.");
	// correct in event of missing or corrupt offset information
	else if(data_offset < diskHeaderInfoSize()) {
		sprintf(msg, "Warning:  corrupt header data offset (%d)", data_offset);
		data_offset = diskHeaderInfoSize();
		retcode = 1;		// nonfatal error
	}
	else retcode = 1;
	if(strlen(msg))
		Application::alert(msg);
	return retcode;
}

const char*
CS_PvocHeader::magicError() {
	static char msg[80];
	sprintf(msg, "Invalid Csound Pvoc file magic number (%d != %d)",
		magic(), PV_MAGIC);
	return msg;
}

int
CS_PvocHeader::writeInfo(DataFile *file) {
	DiskHeader head;
	head.magic = magic();
	head.headBsize = dataOffset();
	head.dataBsize = dataSize();
	head.dataFormat = PVFLOAT;					// ALWAYS, for now
	head.samplingRate = sampleRate();
	head.channels = 1;				// can I get the actual value somewhere?
	head.frameSize = nChans() - 2;	// store fftsize without incl. extra pair
	head.frameIncr = long(sampleRate() / frameRate());
	head.frameBsize = 2 * sizeof(float) * (head.frameSize / 2 + 1);	// ALWAYS
	head.frameFormat = PVPVOC;
	head.minFreq = 0.0;
	head.maxFreq = sampleRate() / 2.0;
	head.freqFormat = PVLIN;
	file->write((void *) &head, diskHeaderInfoSize());
	return file->good();
}
