// soundheader.h

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993, 1994 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       December 13, 1994
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/


// Specialized Header subclasses for reading, parsing, and verifying headers
// on various types of on-disk sound files.

#ifndef SOUNDHEADER_H
#ifdef __GNUG__
#pragma interface
#endif
#define SOUNDHEADER_H

#include "sndconfig.h"
#include "header.h"

#define SIZEOF_BSD_HEADER 1024

class SoundHeader : public Header {
	typedef Header Super;
public:		// class methods
	enum Type { None=0x1, Snd=0x2, Hybrid=0x4, Ircam=0x8, Aifc=0x10, Wav=0x20 };
	enum Chans { MaxChannels = 4 };
	static Type defaultHeaderType() { return default_HeaderType; }
	static void setDefaultHeaderType(Type type) { default_HeaderType = type; }
			// meta-constructors
	static SoundHeader* create(DataFile*, DataType, int, int, double);
	static SoundHeader* create(Type, DataType, int, int, double);
public:		// object methods
	~SoundHeader() {}
	virtual int sampleRate() { return samprate; }
	virtual double peakAmp() { return peakamp; }
	virtual Type headerType() { return header_type; }
protected:
	redefined int bytesPerSecond();
	redefined int checkHeader();
	virtual boolean isValid(DataType)=0;
	SoundHeader(DataType format, int rate, int chans, double peak, int magic)
		: Header(magic, format, chans), samprate(rate), peakamp(peak) {}

	class SoundConfigRequester : public Header::ConfigRequester {
	public:
		SoundConfigRequester(Header* header) : ConfigRequester(header) {}
	protected:
    	redefined void configureRequest(Request &);
 	  	redefined boolean setValues(Request &);
	private:
		redefined const QueryInfo *requestInfo();
		enum SampleType {
			Linear8 = 0x1, MuLaw = 0x2, Linear16 = 0x4, Float = 0x8
		};
	};
	friend boolean SoundConfigRequester::setValues(Request &);
	redefined const char* magicError();
public:
	redefined Requester* configRequester() {
		return new SoundConfigRequester(this);
	}
protected:	// object members
	Type header_type;
	int samprate;
	double peakamp;
private:	// class members
	static Type default_HeaderType;
};

// These next three represent the actual physical layout of the header on
// disk.

struct U_SND_Header {	// for use in union, below (where no ctor allowed)
	int magic;
	int data_loc;
	int data_size;
	int format;
	int samp_rate;
	int nchans;
#if defined(NeXT) || defined(USE_NeXT_HEADERS)
	char info[4];
#endif
};

// for use outside of union

struct SND_Header : public U_SND_Header {
	SND_Header(int m=SND_MAGIC, int dl=0, int ds=0, int fmt=0, int sr=0,
			int nch=0) {
		magic = m; data_loc = dl; data_size = ds;
		format = fmt; samp_rate = sr; nchans = nch;
	}
};

union IRCAM_Header {
	IRCAM_Header(int m=SF_MAGIC, float sr=0, int ch=0, int pm=0) {
		sfinfo.sf_magic = m; sfinfo.sf_srate = sr; sfinfo.sf_chans = ch;
		sfinfo.sf_packmode = pm; *((short *) &sfinfo.sf_codes) = 0;
	}
	struct Info {
		int	  sf_magic;
		float	  sf_srate;
		int	  sf_chans;
		int	  sf_packmode;
		char	  sf_codes;
	} sfinfo;
	char	filler[SIZEOF_BSD_HEADER];
};

union SF_Header {
	struct Info {	
		U_SND_Header sndheader;
		int	  sf_magic;
		float	  sf_srate;
		int	  sf_chans;
		int	  sf_packmode;
		char	  sf_codes;
	} sfinfo;
	char	filler[SIZEOF_BSD_HEADER];
};

// this is a generic header for either NeXT or Sparc files

class SndOrIrcamSoundHeader : public SoundHeader {
protected:
	SndOrIrcamSoundHeader(DataType format, int rate, int chans, double peak,
		int magic) : SoundHeader(format, rate, chans, peak, magic) {}
	redefined boolean isMagic();
	redefined int readInfo(DataFile *);
	redefined int readComment(DataFile *);
	redefined int writeComment(DataFile *);

	virtual int readSndInfo(DataFile *);
	virtual int readIrcamInfo(DataFile *);
	virtual int extractPeak(const char *);
	virtual int extractComment(const char *);
	virtual int loadPeak(const char *);
	virtual int loadComment(const char *);
	redefined boolean isValid(DataType t) {
		return (t == MuLawData || t == ShortData || t == FloatData);
	}
protected:
	enum SF_Code { SF_END = 0, SF_MAXAMP, SF_COMMENT, SF_LINKCODE };
	struct SFCode {
		short	code;
		short	bsize;
	};
	struct SFMaxamp {
		float	value[MaxChannels];
		long	samploc[MaxChannels];
		long	timetag;
	};
	struct SFComment {
		char 	comment[MAXCOMM];
	};
	const char *getSFCode(const char *, int);
	int putSFCode(const char *, const char *, const SFCode *);
};

class IrcamHeader : public SndOrIrcamSoundHeader {
public:
	IrcamHeader(DataType format=ShortData, int rate=defaultSampRate,
		int chans=1, double peak=0);
	virtual ~IrcamHeader() {}
protected:
	IrcamHeader() : SndOrIrcamSoundHeader(ShortData, defaultSampRate, 1, 0,
		SF_MAGIC) {}
	redefined int diskHeaderSize() { return sizeof(IRCAM_Header); }
	redefined int diskHeaderCommentSize() {
		return diskHeaderSize() - sizeof(IRCAM_Header::Info);
	}
	redefined int writeInfo(DataFile *);
};

// NeXT headers have 4 bytes for char comment;  Sun au headers do not

#if defined(USE_NeXT_HEADERS) || defined(NeXT)
#define INFO_SIZE 4
#else
#define INFO_SIZE 0
#endif

class SndHeader : public SndOrIrcamSoundHeader {
public:
	SndHeader(DataType format=ShortData, int rate=defaultSampRate, int chans=1);
	virtual ~SndHeader() {}
protected:
	SndHeader() : SndOrIrcamSoundHeader(ShortData, defaultSampRate, 1, 0,
		SND_MAGIC) {}
	redefined int diskHeaderSize() { return sizeof(SND_Header); }
	redefined int diskHeaderCommentSize() { return INFO_SIZE; }
	redefined int diskHeaderInfoSize() {
		return sizeof(SND_Header) - INFO_SIZE; 
	}
	redefined int writeInfo(DataFile *);
	redefined int writeComment(DataFile *);
	redefined int commentLength();
};

class HybridHeader : public SndHeader {
public:
	HybridHeader(DataType format=ShortData, int rate=defaultSampRate,
		int chans=1, double peak=0);
	virtual ~HybridHeader() {}
protected:
	redefined int diskHeaderSize() { return sizeof(SF_Header); }
	redefined int diskHeaderCommentSize() {
		return diskHeaderSize() - sizeof(SF_Header::Info);
	}
	redefined int writeInfo(DataFile *);
	redefined int writeComment(DataFile *);
};

#endif
