// optionsetter.C

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993, 1994 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       December 13, 1994
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/

#ifdef __GNUG__
#pragma implementation
#endif

#include "application.h"
#include "dialogbox.h"
#include "optionsetter.h"
#include "query.h"
#include "request.h"
#include "lpcdata.h"
#include "sound.h"
#include "soundheader.h"

const QueryInfo *
GlobalOptionSetter::requestInfo() {
	static QueryLabelInfo labelInfo[] = {
		{ "Set Global Program Options: " },
		{ nil }
	};      
	static QueryInfo info[] = {
		{ labelInfo, "", nil },
		{ nil }
	};
	return info;
}

void
GlobalOptionSetter::configureRequest(Request &request) {
	request.appendChoice("Auto-Place Windows on Screen: ", "|Yes|No|",
		Application::globalResourceIsTrue("AutoPlaceWindows") ? 0x1 : 0x2, true);
	request.appendChoice("Dialog Panels Ignore Window Manager: ", "|Yes|No|",
		DialogBox::overrideWindowManager() ? 0x1 : 0x2, true);
}

boolean
GlobalOptionSetter::setValues(Request &request) {
	QueryChoice choices[2];
	request.retrieveChoices(choices, 2);
	Application::setGlobalResource(
		"AutoPlaceWindows", (choices[0] == 0x1) ? "true" : "false"
	);
	DialogBox::overrideWindowManager(choices[1] == 0x1);
	return true;
}

//********

const QueryInfo *
FileOptionSetter::requestInfo() {
	static QueryLabelInfo labelInfo[] = {
		{ "Set File Options: " },
		{ nil }
	};      
	static QueryInfo info[] = {
		{ labelInfo, "", nil },
		{ nil }
	};
	return info;
}

void
FileOptionSetter::configureRequest(Request &request) {
	request.appendChoice("Read Raw (Headerless) Files: ", "|Yes|No|",
		Application::globalResourceIsTrue("ReadRawFiles") ? 0x1 : 0x2, true);
	request.appendChoice("Store/Recall Browser Path: ", "|Yes|No|",
		Application::globalResourceIsTrue("BrowserUseLastPath") ? 0x1 : 0x2, true);
}

boolean
FileOptionSetter::setValues(Request &request) {
	const int nChoices = 2;
	QueryChoice c[nChoices];
	request.retrieveChoices(c, nChoices);
	Application::setGlobalResource(
		"ReadRawFiles", (c[0] == 0x1) ? "true" : "false"
	);
	Application::setGlobalResource(
		"BrowserUseLastPath", (c[2] == 0x1) ? "true" : "false"
	);
	return true;
}

//********

const QueryInfo *
SoundOptionSetter::requestInfo() {
	static QueryLabelInfo labelInfo[] = {
		{ "Set Sound Options: " },
		{ nil }
	};      
	static QueryInfo info[] = {
		{ labelInfo, "", nil },
		{ nil }
	};
	return info;
}

void
SoundOptionSetter::configureRequest(Request &request) {
	request.appendValue("Default Sampling Rate: ", Sound::defaultSampleRate(),
		CharCheck::posIntsOnly);
	request.appendChoice("Default Sound Sample Format: ",
		"|8-bit linear|8-bit mu-law|16-bit linear|32-bit floating point|",
		linearEnumToPowerOfTwo(Sound::defaultDataType()),
		true
	);
	request.appendChoice("Default Sound Header Format: ",
		"|Raw (No Header)|Snd/au|Hybrid|BSD/IRCAM|AIF-C|WAV|",
		SoundHeader::defaultHeaderType(),
		true
	);
}

boolean
SoundOptionSetter::setValues(Request &request) {
	QueryValue srate;
	request.retrieveValues(srate);
	Sound::setDefaultSampleRate(srate);
	const int nChoices = 2;
	QueryChoice c[nChoices];
	request.retrieveChoices(c, nChoices);
	int state = c[0];
	DataType format = (state == 0x1) ? CharData : (state == 0x2) ? MuLawData
		: (state == 0x4) ? ShortData : FloatData;
	Sound::setDefaultDataType(format);
	SoundHeader::setDefaultHeaderType(SoundHeader::Type(c[1].state()));
	return true;
}

//********

const QueryInfo *
LPCOptionSetter::requestInfo() {
	static QueryLabelInfo labelInfo[] = {
		{ "Set LPC Data Options: " },
		{ nil }
	};      
	static QueryInfo info[] = {
		{ labelInfo, "Default samp rate = default sound samp rate.", nil },
		{ nil }
	};
	return info;
}

void
LPCOptionSetter::configureRequest(Request &request) {
	request.appendValue("Default Frame Rate: ", LPCData::defaultFrameRate(),
		CharCheck::posNumsOnly);
	request.appendValue("Default Number of Filter Poles: ",
		LPCData::defaultNumberOfPoles(), CharCheck::posIntsOnly);
	request.appendChoice("Default LPC Header Format: ",
		"|Raw (No Header)|Csound-format Header|",
		LPCHeader::defaultHeaderType(),
		true
	);
}

boolean
LPCOptionSetter::setValues(Request &request) {
	const int nValues = 2;
	QueryValue values[nValues];
	request.retrieveValues(values, nValues);
	LPCData::setDefaultFrameRate(values[0]);
	LPCData::setDefaultNumberOfPoles(values[1]);
	QueryChoice headerFormat;
	request.retrieveChoices(headerFormat);
	LPCHeader::setDefaultHeaderType(LPCHeader::Type(headerFormat.state()));
	return true;
}

