// interpolater.h

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993, 1994 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       December 13, 1994
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/


// This class and its closely-related subclass Transposer perform a second-
// order parabolic interpolation upon the input, storing the result into the
// output.  Note that this class, as with all InputOutputFunction subclasses,
// does not directly modify its input.

#ifndef INTERPOLATER_H
#ifdef __GNUG__
#pragma interface
#endif
#define INTERPOLATER_H

#include "localdefs.h"
#include "iofunction.h"

struct QueryInfo;

class Interpolater : public InputOutputFunction {
	typedef InputOutputFunction Super;
public:
	Interpolater(Data* input, Data* output, double factor, boolean filt=false);
	Interpolater(Data* input, Data* output, int newlen, boolean filt=false);
	Interpolater(Data* input, Data* output);
	virtual ~Interpolater();
	static Modifier* create(class DataEditor *);
	redefined void restoreState();
	redefined const char* message() { return "Interpolating..."; }
protected:
	redefined void configureRequest(Request &);
	redefined int newLength(int oldLength);
	redefined boolean areCompatibleLengths(int, int);
	redefined void initialize();
	redefined boolean setValues(class Request &r);
	double preProcess(double input);
	double postProcess(double input);
	redefined int processInput(boolean flushing);
	virtual double interpFactor() { return myInterpFactor; }
	void setInterpFactor(double factor) { myInterpFactor = factor; }
	void setFilter(boolean flag) { useFilter = flag; }
	void setGain(double gn) { gain = gn; }
	void increment() { incount++; }
	boolean filtering() { return (processMode != None); }
	enum QLen { InputQLen = 1024, OutputQLen = 1024 };
	enum TransposeMode { Ratio = 0x1, OctavePoint = 0x2, LinearOctave = 0x4 };
	enum ProcessMode { None, Pre, Post } processMode;
private:
	redefined const QueryInfo *requestInfo();
	void clear();
	static double interpolate(double, double, double, double fraction);
private:
	double myInterpFactor;
	double gain;
	double voldsig;
	double oldsig;
	double sig;
	int incount;
	double fractionCounter;
	boolean useFilter;
	SimpleFunction* filter;
};

// note:  this will need to be carefully redefined in variable Interpolater

inline int
Interpolater::newLength(int oldLength) {
	return (myInterpFactor != 0.0) ? roundUp(oldLength/myInterpFactor) : 0;
}

inline void
Interpolater::clear() {
	voldsig = oldsig = sig = 0; incount = -1; fractionCounter = 0;
}

//********

class Transposer : public Interpolater {
public:
	Transposer(Data* input, Data* output) : Interpolater(input, output) {}
	static Modifier* create(class DataEditor *);
	redefined const char* message() {
		return filtering() ? "Transposing and filtering..." : "Transposing...";
	}
protected:
	redefined void configureRequest(Request &) {} // redefined to nothing
	redefined boolean setValues(class Request &r);
private:
	redefined const QueryInfo *requestInfo();
};

#endif
