// header.C

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993, 1994 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       December 13, 1994
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/


#ifdef __GNUG__
#pragma implementation
#endif

#include "localdefs.h"
#include "application.h"
#include "comment.h"
#include "datafile.h"
#include "header.h"
#include "typeconvert.h"

const int Header::defaultSampRate = 44100;

Header::~Header() { Resource::unref(comment); }

boolean
Header::configure(Controller *c) {
	boolean status = false;
	Requester* requester = configRequester();
	status = (requester != nil && requester->configure(c));
	delete requester;
	return status;
}

void
Header::initialize() {
	magic_number = 0;
	data_offset = 0;
	data_type = NoData;
	data_size = 0;
	nchans = 0;
}

void
Header::setComment(const char *com) {
	setComment(new Comment(com));
}

void
Header::setComment(Comment *c) {
	if(c != comment) {
		Resource::unref(comment);
		comment = c;
		if(comment != nil) {
			comment->ref();
		}
	}
}

int
Header::commentLength() {
	return (comment != nil) ? comment->len() : 0;
}

// this routine reads the magic number from the file and then checks it for
// validity.  If raw file reading is enabled, it fails silently so that the
// user may be queried for raw file reading options

int
Header::checkMagic(DataFile* file) {
	boolean status = true;
	if ((status = readMagic(file)) == true) {
		if (!isMagic()) {
			if (!Application::globalResourceIsTrue("ReadRawFiles"))
				Application::alert(magicError());
			status = false;
		}
	}
	return status;

}

// static public method

int
Header::readMagicNumber(DataFile *file) {
	int magic = -1;
	file->read((void *) &magic, sizeof(magic));
	file->clear();		// no file failure here because this is used to check
	file->seek(0);		// files that may not yet exist
	return magic;
}

int
Header::readMagic(DataFile *file) {
	if((magic_number = Header::readMagicNumber(file)) == -1) {
		Application::alert("Header::readMagic:  file read error.");
		return false;
	}
	return true;
}

// compares the endian attribute of the header class
// with the Application (machine architecture) endian attribute

boolean
Header::needsByteSwapping() {
	return isLittleEndian() != Application::isLittleEndian();
}

int
Header::read(DataFile *file) {
	// default operations for header
	int retcode = false;
	if(isRaw())
		retcode = checkHeader();
	else {
		initialize();
		file->setBytesSwapped(needsByteSwapping());
		if(file->readable()) {
			if(checkMagic(file) == true)
				retcode = (readInfo(file) && readComment(file));
		}
	}
	if(retcode) {
		// adjust data size and offset to account for inskip and duration
		file->setHeaderSize(dataOffset());	// header size in bytes
		int skipOffset = int(file->skipTime() * bytesPerSecond());
		file->setOffset(skipOffset);
		file->setReadSize(int(file->duration() * bytesPerSecond()));
		setDataSize(file->readSize());
		// seek to beginning of desired data
		retcode = seekInFile(file, skipOffset);
	}
	else
		file->failif(true);		// to notify calling routines
	return retcode;
}

int
Header::seekInFile(DataFile* file, int offset) {
	if(!file->seek(dataOffset() + offset).good()) {
		Application::error("seekInFile: seek error:");
		return false;
	}
	return true;
}

int
Header::readComment(DataFile *file) {
	int textSize = dataOffset() - diskHeaderInfoSize();
	if(textSize > 0) {
		// seek to beginning of text information
		file->seek(diskHeaderInfoSize());
		char *com = new char[textSize];
		file->read((void *) com, textSize);
		com[textSize-1] = '\0';		// null terminate
		setComment(com);
		delete [] com;
	}
	return file->good();
}

int
Header::write(DataFile *file) {
	int status = false;
	file->setBytesSwapped(needsByteSwapping());
	if(checkHeader() == true) {
		file->reOpen("w+");				// to truncate file
		file->seek(0);
		if(isRaw())						// dont check any further if raw
			status = true;
		else if(file->writable()) {
			file->setHeaderSize(dataOffset());
			if(writeInfo(file)) {
				if(writeComment(file))
					status = true;
			}
			if(status == false)
				file->error();
		}
		else {
			char msg[256];
			sprintf(msg, "Header::write: %s is not writable.", file->name());
			Application::alert(msg);
		}
	}
	return status;
}

// default method writes 4 null chars

int
Header::writeComment(DataFile *file) {
	static char text[4];
	file->write((void *) text, sizeof(text));
	return file->good();
}

int
Header::sampleSize() { return type_to_sampsize(dataType()); }

// LPC Header Methods

LPCHeader::Type LPCHeader::default_HeaderType = LPCHeader::With;

LPCHeader::LPCHeader(int poles, double framert, int srate, double duration)
	: FrameDataHeader(LP_MAGIC, FloatData, poles+4, framert, srate, duration),
		npoles(poles) {
	if(defaultHeaderType() == None)
		setRaw();
}

// static public method

int
LPCHeader::readMagicNumber(DataFile *file) {
	int magic = -1;
	int temp[2];	// LPC magic is second int in header
	if(file->read((void *) temp, sizeof(temp)).good())
		magic = temp[1];
	file->clear();		// no file failure here because this is used to check
	file->seek(0);		// files that may not yet exist
	return magic;		// -1 indicated bad read
}

int
LPCHeader::readMagic(DataFile *file) {
	if((magic_number = LPCHeader::readMagicNumber(file)) == -1) {
		Application::alert("LPCHeader::readMagic:  file read error.");
		return false;
	}
	return true;
}

const char*
LPCHeader::magicError() {
	static char msg[80];
	sprintf(msg, "Invalid LPC file magic number (%d != %d)", magic(), LP_MAGIC);
	return msg;
}

int
LPCHeader::readInfo(DataFile *file) {
	DiskHeader head;
	int status = false;
	if(file->read((void *) &head, diskHeaderSize()).good()) {
		samprate = int(head.srate);
		npoles = head.npoles;
		dur = head.duration;
		framerate = head.framrate;
		setDataOffset(head.headersize);
		nchans = head.nvals;
		data_type = FloatData; // since other progs create it this way
		status = true;
	}
	else
		Application::alert("LPCHeader::readInfo:  file read error.");
	return status && checkHeader();
}

int
LPCHeader::checkHeader() {
	char msg[64];
	int retcode = 0;
	if(npoles > maxPoles || npoles != (nchans - 4))
		sprintf(msg, "Bad header: npoles= %d", npoles);
	else if(samprate < 4000 || samprate > 64000)
		sprintf(msg, "Bad header: samp rate= %d", samprate);
	else if(data_type != FloatData)
		sprintf(msg, "Bad header: unknown LPC data type: %d",
			data_type);
	else retcode = 1;
	if(retcode != 1)
		Application::alert(msg);
	return retcode;
}

int
LPCHeader::writeInfo(DataFile *file) {
	DiskHeader head(diskHeaderSize(), magic_number, npoles, nchans,
		framerate, samprate, dur);
	file->write((void *) &head, diskHeaderInfoSize());
	return file->good();
}

// Pitch Track Header Methods

int
PCHHeader::readMagic(DataFile *file) {
	file->seek(0);
	return file->good();
}

int
PCHHeader::readInfo(DataFile *file) {
	Super::readInfo(file);
	nchans = 2;
	data_type = FloatData;	// since other progs create it this way
	return true;		// no header on pitch files
}

// FFT Header Methods

const char*
FFTHeader::magicError() {
	static char msg[80];
	sprintf(msg, "Invalid FFT file magic number (%d != %d)", magic(), FFT_MAGIC);
	return msg;
}

int
FFTHeader::readInfo(DataFile *file) {
	DiskHeader head;
	int status = false;
	if(file->read((void *) &head, diskHeaderSize()).good()) {
		samprate = int(head.srate);
		nchans = head.npoints;
		setDataOffset(head.headersize);
		data_type = DoubleData;		// no way to check -- we assume this
		status = true;
	}
	else
		Application::alert("FFTHeader::readInfo:  file read error.");
	return status && checkHeader();
}

int
FFTHeader::checkHeader() {
	int status = false;
	char msg[128];
	if(nChans() % 2 != 0)
		sprintf(msg, "Bad number of FFT points: %d [must be power of 2]",
			nChans());
	else status = true;
	if(status != true)
		Application::alert(msg);
	return status;
}

int
FFTHeader::writeInfo(DataFile *file) {
	DiskHeader head(magic_number, diskHeaderSize(), nPoints(), sampleRate());
	file->write((void *) &head, diskHeaderInfoSize());
	return file->good();
}

// Envelope Header Methods

EnvelopeHeader::Type EnvelopeHeader::default_HeaderType = EnvelopeHeader::With;

const char*
EnvelopeHeader::magicError() {
	static char msg[80];
	sprintf(msg, "Invalid Envelope file magic number (%d != %d)",
		magic(), EVP_MAGIC);
	return msg;
}

int
EnvelopeHeader::readInfo(DataFile *file) {
	DiskHeader head;
	int status = false;
	if(file->read((void *) &head, sizeof(head)).good()) {
		samprate = int(head.srate);
		npoints = head.npoints;
		setDataOffset(head.headersize);
		nchans = 1;
		data_type = FloatData;
		status = true;
	}
	else
		Application::alert("EnvelopeHeader::readInfo:  file read error.");
	return status && checkHeader();
}

int
EnvelopeHeader::checkHeader() {
	int status = false;
	if(data_type != FloatData)
		Application::alert("Bad data type [must be floating point].");
	else if(nchans != 1)
		Application::alert("Bad header:  nchans must be 1.");
	else status = true;
	return status;
}

int
EnvelopeHeader::writeInfo(DataFile *file) {
	DiskHeader head(magic_number, diskHeaderSize(), npoints, samprate);
	file->write((void *) &head, diskHeaderInfoSize());
	return file->good();
}
